% finaquant Financial Analytics - www.finaquant.com
% Copyright: Tunc Ali Ktkcoglu 2012, version: 24April2012
% TITLE: Share Class Fee Calculator
% Related web page:
% http://finaquant.com/fee-calculator-and-analyzer-for-share-classes-of-mutual-funds-1-calculation/991
%
% FEE TYPES:
% Front-end load with possible volume discounts
% Back-end load with DCSC (Contingent Deferred Sales Charges)
% Annual distribution fees (12b-1) in %
% Annual management fees with possible volume discounts
% Other annual service fees in % and/or absolute amount
% Other subscription fees in % and/or absolute amount
% Other redemption fees in % and/or absolute amount

%**************************************************************************
% INPUT PARAMETERS
%**************************************************************************
disp('Set input parameters')
FrondEndLoadRates = [       % percentage and exclusive upper limit
    5.00        50000;      % fee rate = 5.00% for 0 <= Volume Amount < 50000
    4.50        100000;     % fee rate = 4.50% for 50000 <= Volume Amount < 100000
    3.70        250000;
    2.80        500000;
    2.00        999999999];     % fee rate = 2% for amounts above 50000
% Set FrondEndLoad = [0  999999999] if there is no front-end load
    
BackEndLoadCDSCRates = [
    5.00                % i.e. 5% deducted if shares are redeemed within 1st year
    4.00                % i.e. 4% deducted if shares are redeemed within 2nd year
    3.00
    2.00
    1.00 ];             % no deductions after 6th year
% Set BackEndLoadCDSC = 0; if there is no back-end load with CDSC

AnnualManagementFeeRates = [   % fee rates in BP (basis points, 1 BP = 0.01% = 1/100000)
    80          50000;         % fee rate = 80 BP for 0 <= Volume Amount < 500000
    70          100000;        % fee rate = 70 BP for 500000 <= Volume Amount < 1000000
    60          999999999];    % fee rate = 60 BP for Volume Amount >= 1000000
% Set ManagementFeeRates = [x 999999999]; if there is a single
% fee rate without volume discounts

AnnualDistributionFeeRate = 0.5; % i.e. 0.5% annually

% Fee scale calculation logic for management fees
MngFeeScaleLogic = 'class';
% class: There is a single fee rate applied on the whole volume
% f.e. volume = $600000 --> MngFee = $600000 x 70/10000 w.r.t. given
% tiered rates above in AnnualManagementFeeRates
%
% level: Each volume range is multiplied with its corresponding fee rate
% f.e. volume = $600000 --> MngFee = $500000 x 80/10000 + $100000 x 70/10000

OtherAnnualServiceFeeRates = [10 0];    % 10 BP + $0 per share, annually 

OtherSubscriptionFeeRates = [0 0];      % 0% + $0 per share, at subscription

OtherRedemptionFeeRates = [0 0];        % 0% + $0 per share, at redemption
    
% Total number of periods (years) for calculation
N = 20;

% Assumed fund return for all years before all fee costs
AssumedFixedFundReturn = 5;     % i.e. 5%

% Fund returns per each year within the calculation time span (i.e. N years)
AssumedFundReturns = AssumedFixedFundReturn * ones(1,N);
% set AssumedFundReturns = [x1 x2 x3 ... xN] if you want to give assumed
% fund returns for each period (year) seperately

% Number of fund shares purchased (#subscriptions at the beginning)
NumberOfShares = 500; 

% Gross price of a share before all costs at the beginning 
SharePrice = 100;

%**************************************************************************
% all input parameters are ready: call multi-period fee function
%**************************************************************************
disp('Call multi-period fee calculation function')
[ResultMatrix,     ...     % stores all intermediate results, a row for each period (year)
ResultMatrixFields,     ...     % field (column) names of result matrix in a cell array
EffectiveFundReturns,   ...     % effective fund returns for each period after all fees
EffectiveAssetValue]    ...     % effective total asset values for each period after all fees
    = FeeCalculator_MultiPeriod(    ...
    FrondEndLoadRates,              ...
    BackEndLoadCDSCRates,           ...
    AnnualDistributionFeeRate,      ...
    AnnualManagementFeeRates,       ...
    MngFeeScaleLogic,               ...
    OtherAnnualServiceFeeRates,     ...
    OtherSubscriptionFeeRates,      ...
    OtherRedemptionFeeRates,        ...
    AssumedFundReturns,             ...
    SharePrice,                     ...     % gross subscription price
    NumberOfShares,                 ...     % number of shares
    N);                                     % total number of periods (years, quarters etc.)

% total returns in percentage
InitialInvestment = NumberOfShares * SharePrice;
TotalFundReturns = (EffectiveAssetValue - InitialInvestment) / InitialInvestment * 100;

% construct result table with total returns and final asset values at each period-end
AddedColumnNames = {'EffReturnPeriod','TotalReturn','FinalAssetValue'};
AddedTable = [AddedColumnNames; num2cell([EffectiveFundReturns', TotalFundReturns', EffectiveAssetValue'])];

ResultTable = [ResultMatrixFields; num2cell(ResultMatrix)];
ResultTable = [ResultTable, AddedTable]

%**************************************************************************
% write results into an excel file in the same working directory
%**************************************************************************
disp('Write results into an excel sheet')
xlswrite('results_matlab.xls',ResultTable,'result1')